package org.biohackathon.SPARQLBuilder.OWL;

/**
 * クエリを生成するための機能を提供する核インターフェイス
 * @author YAMAGUCHI
 * @author GO
 * @author KOZAKI
 * @author KOBAYASHI
 * @since 28.01.2014
 * @version 29.01.2014
 */
public interface OWLQueryBuilder {
	/**
	 * 明示的にRDFで書かれているクラスを取得する
	 * <p>
	 * 指定されたgraphURIsの中から、キーワードにヒットするラベルを持つクラス（rdfs:Class）をすべて返す
	 * <br>
	 * </p>
	 * 
	 * @param graphURIs　検索対象のgraphのURI配列 (nullや長さ0の配列も可)
	 * @param keyword （nullや空文字は不可）
	 * @return クラスURIの配列
	 * @throws Exception
	 * @since 28.01.2014
	 */
	public SClass[] getOWLClasses(String[] graphURIs, String keyword) throws Exception;

	/**
	 *　インスタンスを取得する
	 * <p>
	 * 指定されたgraphURIsの中から、キーワードにヒットするラベルを持つインスタンスをすべて返す
	 * <br>
	 * ここでインスタンスとは、rdf:typeの主語として記述されているものをいう
	 * <br>
	 * </p>
	 * 
	 * @param graphURIs　検索対象のgraphのURI配列 (nullや長さ0の配列も可)
	 * @param keyword （nullや空文字は不可）
	 * @return クラスの配列
	 * @throws Exception
	 * @since 28.01.2014
	 */
	public Instance[] getInstances(String[] graphURIs, String keyword) throws Exception;

	/**
	 *　指定されたクラスを起点とし、明示的に記述されているOWLのproperty制約を調べ、そのproperty制約で
	 * 関連づけられているクラスを網羅的に取得する
	 * <p>
	 * 処理対象データをgraphURIsで指定することができる
	 * <br>
	 * <br>
	 * </p>
	 * 
	 * @param graphURIs　検索対象のgraphのURI配列 　(nullや長さ0の配列も可)
	 * @param originClass 起点となるクラスのURI　（null不可）
	 * @param limit 解として返されるClassLink配列の長さの上限値を指定する。(上限値を設定しない場合は0以下の値を指定する)
	 * @return ClassLinkの配列。ここには、取得されたクラスと関係づけられているプロパティ、関係の向きが含まれる。
	 * @throws Exception
	 * @since 28.01.2014
	 */
	public ClassLink[] getNextClass(String[] graphURIs, String originClass, int limit, boolean countLinks) throws Exception;

	/**
	 *　指定されたクラスを起点とし、そのクラスに属しているインスタンスとリンクが張られているインスタンスの集合を取得し、取得したインスタンスのクラスを網羅的に取得する
	 * <p>
	 * ここでインスタンスとは、rdf:typeの主語として記述されているものをいう
	 * <br>
	 * 処理対象データをgraphURIsで指定することができる
	 * <br>
	 * </p>
	 * 
	 * @param graphURIs　検索対象のgraphのURI配列 　(nullや長さ0の配列も可)
	 * @param originClass 起点となるクラスのURI　（null不可）
	 * @param limit 解として返されるClassLink配列の長さの上限値を指定する。(上限値を設定しない場合は0以下の値を指定する)
	 * @param countLinks インスタンス間のリンクの数（トリプル数）を返したいときはtrueを指定する
	 * @return ClassLinkの配列。ここには、取得されたクラスと関係づけられているプロパティ、関係の向きが含まれる。
	 * @throws Exception
	 * @since 28.01.2014
	 */
	public ClassLink[] getNextClassViaInstanceLink(String[] graphURIs, String originClass, int limit) throws Exception;

	public Path[] getPaths(String startClass, String endClass) throws Exception;
	public String createSPARQL(Path path) throws Exception;

	/**
	 *　指定されたインスタンスを起点とし、そのインスタンスにリンクが張られているインスタンスの集合を取得する。
	 *　取得された各インスタンスのクラスも網羅的に取得する
	 * <p>
	 * ここでインスタンスとは、rdf:typeの主語として記述されているものをいう
	 * <br>
	 * 処理対象データをgraphURIsで指定することができる
	 * <br>
	 * </p>
	 * 
	 * @param graphURIs　検索対象のgraphのURI配列 　(nullや長さ0の配列も可)
	 * @param originInstance 起点となるインスタンスのURI　（null不可）
	 * @param limit 解として返されるInstanceLink配列の長さの上限値を指定する。(上限値を設定しない場合は0以下の値を指定する)
	 * @return InstanceLinkの配列。ここには、取得されたクラスと関係づけられているプロパティ、関係の向きが含まれる。
	 * @throws Exception
	 * @since 28.01.2014
	 */
 InstanceLink[] getNextInstancesViaInstanceLink(String[] graphURIs, String originInstance,
			int limit) throws Exception;
	public LabelMap[] getLabels(String[] graphURIs, String[] resourceURIs, String language) throws Exception;

}
