package org.biohackathon.SPARQLBuilder.OWL;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.HttpClient;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpPost;
//import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;

import com.hp.hpl.jena.query.Query;
import com.hp.hpl.jena.query.QueryExecution;
import com.hp.hpl.jena.query.QueryExecutionFactory;
import com.hp.hpl.jena.query.QueryFactory;
import com.hp.hpl.jena.query.QuerySolution;
import com.hp.hpl.jena.query.ResultSet;
import com.hp.hpl.jena.rdf.model.Resource;

/**
 * クエリを生成するための機能を提供する核クラス
 * @author Norio KOBAYASHI
 * @since 28.01.2014
 * @version 29.01.2014
 */
public class OWLQueryBuilderImpl implements OWLQueryBuilder {

	// private Model model = null;
	private String endpointURI = null;

	
	/**
	 * アクセスするSPARQL endpointのURIを指定する構成子
	 * 
	 * @param endpointURI　アクセスするSPARQL endpointのURI
	 * @throws Exception
	 * @since 28.01.2014
	 */
	public OWLQueryBuilderImpl(String endpointURI) {
		this.endpointURI = endpointURI;
	}

	/**
	 * テストに使用するmainメソッド
	 * <p>
	 * クエリビルダーの本番プログラムではこのメソッドは使用しない
	 * </p>
	 * @param args 使用しない
	 * @throws Exception
	 * @since 28.01.2014
	 */
	public static void main(String[] args) throws Exception {
		String sparqlEndpoint = "http://dbpedia.org/sparql";
		String keyword = "artiste";
		String[] graphURIs = new String[0];

		OWLQueryBuilder builder = new OWLQueryBuilderImpl(sparqlEndpoint);
		String[] clsURIs = builder.getOWLClasses(graphURIs, keyword);
		for(String cls: clsURIs){
			System.out.println(cls);
		}

		System.out.println("CLS");
		ClassLink[] cls = builder.getNextClass(null, clsURIs[0], 0);
		if( cls != null ){
			for( ClassLink cl: cls){
				System.out.println(cl.toString());
			}
		}

		System.out.println("CLS-INS");
		cls = builder.getNextClassViaInstanceLink(null, clsURIs[0], 100);
		if( cls != null ){
			for( ClassLink cl: cls){
				System.out.println(cl.toString());
			}
		}

		System.out.println("Instances");
		Instance[] ins = builder.getInstances(null, "\"A.C. Reed\"@en");
		if( ins != null ){
			for( Instance in: ins){
				System.out.println(in.toString());
			}
		}

		
		System.out.println("INS-INS");
		ins = builder.getInstances(null, "\"A.C. Reed\"@en");
		InstanceLink[] iLinks = builder.getNextInstancesViaInstanceLink(null, ins[0].getInstanceURI(), 100);
		

		
		if( iLinks != null ){
			for( InstanceLink in: iLinks){
				System.out.println(in.toString());
			}
		}
		
		
	}

	
	/**
	 * 明示的にRDFで書かれているクラスを取得する
	 * <p>
	 * 指定されたgraphURIsの中から、キーワードにヒットするラベルを持つクラス（rdfs:Class）をすべて返す
	 * <br>
	 * </p>
	 * 
	 * @param graphURIs　検索対象のgraphのURI配列 (nullや長さ0の配列も可)
	 * @param keyword （nullや空文字は不可）
	 * @return クラスURIの配列
	 * @throws Exception
	 * @since 28.01.2014
	 */
	public String[] getOWLClasses(String[] graphURIs, String keyword)
			throws Exception {
		StringBuffer queryStr = new StringBuffer();
		queryStr.append("PREFIX owl: <http://www.w3.org/2002/07/owl#>\n");
		queryStr.append("PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>\n");
		queryStr.append("PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#>\n");
		queryStr.append("SELECT DISTINCT ?c \n");
		if (graphURIs != null) {
			for (String graphURI : graphURIs) {
				queryStr.append("FROM <");
				queryStr.append(graphURI);
				queryStr.append(">\n");
			}
		}
		queryStr.append("WHERE{\n");
		
		queryStr.append(" {?c rdf:type rdfs:Class} UNION {?c rdf:type owl:Class}\n");

		queryStr.append("      ?c rdfs:label ?label.\n");

//		queryStr.append("      ?c rdfs:label ");
//		queryStr.append(keyword);
//		queryStr.append(".");

		queryStr.append("  FILTER (\n"); 
		queryStr.append("    REGEX( ?label , \"");
		queryStr.append(keyword);
		queryStr.append("\" , \"i\" )\n");
		queryStr.append("  )"); 
		queryStr.append("\n}");

 System.out.println(queryStr.toString());
		
		Query query = QueryFactory.create(queryStr.toString());
		QueryExecution qexec = QueryExecutionFactory.sparqlService(endpointURI,
				query);

		ResultSet results = qexec.execSelect();
		ArrayList<String> solStrings = new ArrayList<String>();
		for (; results.hasNext();) {
			QuerySolution sol = results.next();
			Resource res = sol.getResource("c");
			String uri = res.getURI();
			solStrings.add(uri);
		}
		qexec.close();
		return solStrings.toArray(new String[0]);
	}


	/**
	 *　インスタンスを取得する
	 * <p>
	 * 指定されたgraphURIsの中から、キーワードにヒットするラベルを持つインスタンスをすべて返す
	 * <br>
	 * ここでインスタンスとは、rdf:typeの主語として記述されているものをいう
	 * <br>
	 * </p>
	 * 
	 * @param graphURIs　検索対象のgraphのURI配列 (nullや長さ0の配列も可)
	 * @param keyword （nullや空文字は不可）
	 * @return クラスURIの配列
	 * @throws Exception
	 * @since 28.01.2014
	 */
	public Instance[] getInstances(String[] graphURIs, String keyword)
			throws Exception {
		StringBuffer queryStr = new StringBuffer();
		queryStr.append("PREFIX owl: <http://www.w3.org/2002/07/owl#>\n");
		queryStr.append("PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>\n");
		queryStr.append("PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#>\n");
		queryStr.append("SELECT DISTINCT ?ins ?c \n");
		if (graphURIs != null) {
			for (String graphURI : graphURIs) {
				queryStr.append("FROM <");
				queryStr.append(graphURI);
				queryStr.append(">\n");
			}
		}
		queryStr.append("WHERE{\n");
//		queryStr.append(" {?c rdf:type rdfs:Class} UNION {?c rdf:type owl:Class}\n");

		
		queryStr.append("      ?ins rdf:type ?c.\n");
		queryStr.append("      ?ins rdfs:label ");
		queryStr.append(keyword);
		queryStr.append(".\n");
		queryStr.append(" FILTER (?c != rdf:Property)");
		queryStr.append("}");
				
// System.out.println(queryStr.toString());
		
		Query query = QueryFactory.create(queryStr.toString());
		QueryExecution qexec = QueryExecutionFactory.sparqlService(endpointURI,
				query);

		ResultSet results = qexec.execSelect();
		HashMap<String, HashSet<String>> instanceMap = new HashMap<String, HashSet<String>>();
		for (; results.hasNext();) {
			QuerySolution sol = results.next();
			Resource cls = sol.getResource("c");
			Resource ins = sol.getResource("ins");
			String clsURI = cls.getURI();
			String insURI = ins.getURI();
			if( instanceMap.containsKey(insURI)){
				HashSet<String> classes = instanceMap.get(insURI);
				classes.add(clsURI);
			}else{
				HashSet<String> classes = new HashSet<String>();
				instanceMap.put(insURI, classes);
				classes.add(clsURI);
			}
		}
		qexec.close();
		Set<String> keySet = instanceMap.keySet();
		ArrayList<Instance> instanceList = new ArrayList<Instance>();
		for(String key: keySet){
			Instance ins = new Instance(key, instanceMap.get(key).toArray(new String[0]));
			instanceList.add(ins);
		}
		return instanceList.toArray(new Instance[0]);
	}

	
	/**
	 *　指定されたクラスを起点とし、明示的に記述されているOWLのproperty制約を調べ、そのproperty制約で
	 * 関連づけられているクラスを網羅的に取得する
	 * <p>
	 * 処理対象データをgraphURIsで指定することができる
	 * <br>
	 * <br>
	 * </p>
	 * 
	 * @param graphURIs　検索対象のgraphのURI配列 　(nullや長さ0の配列も可)
	 * @param originClass 起点となるクラスのURI　（null不可）
	 * @param limit 解として返されるClassLink配列の長さの上限値を指定する。(上限値を設定しない場合は0以下の値を指定する)
	 * @return ClassLinkの配列。ここには、取得されたクラスと関係づけられているプロパティ、関係の向きが含まれる。
	 * @throws Exception
	 * @since 28.01.2014
	 */
	public ClassLink[] getNextClass(String[] graphURIs, String originClass,
			int limit) throws Exception {
		StringBuffer queryStr = new StringBuffer();
		queryStr.append("PREFIX owl: <http://www.w3.org/2002/07/owl#>\n");
		queryStr.append("PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>\n");
		queryStr.append("PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#>\n");
		queryStr.append("SELECT DISTINCT ?p ?pDirection ?c (COUNT(?s) AS ?numOfLinks) \n");
		if (graphURIs != null) {
			for (String graphURI : graphURIs) {
				queryStr.append("FROM <");
				queryStr.append(graphURI);
				queryStr.append(">\n");
			}
		}
		queryStr.append("WHERE{\n");
		queryStr.append("  { ?c rdf:type rdfs:Class. }\n  UNION\n  { ?c rdf:type owl:Class. }\n");
		queryStr.append("  { ?p rdfs:domain <");
		queryStr.append(originClass);
		queryStr.append(">.\n    ?p rdfs:range ?c.\n    ?o ?p ?s.}\n  UNION\n");
		queryStr.append("  { ?p rdfs:range <");
		queryStr.append(originClass);
		queryStr.append(">.\n");
		queryStr.append("    ?p rdfs:domain ?c.\n");
		queryStr.append("    ?s ?p ?o.}\n");

		queryStr.append("    ?p ?pDirection ?c.\n");

		
		queryStr.append("  ?s rdf:type ?c.\n");
		queryStr.append("  ?o rdf:type <");
		queryStr.append(originClass);
		queryStr.append(">.\n");
		queryStr.append("}\nGROUP BY ?p ?pDirection ?c\n");
		if (limit > 0) {
			queryStr.append("limit ");
			queryStr.append(limit);
			queryStr.append("\n");
		}
		
		System.out.println(queryStr.toString());	
	
		Query query = QueryFactory.create(queryStr.toString());
		QueryExecution qexec = QueryExecutionFactory.sparqlService(endpointURI,
				query);

		ResultSet results = qexec.execSelect();
		ArrayList<ClassLink> solCLs = new ArrayList<ClassLink>();
		for (; results.hasNext();) {
			QuerySolution sol = results.next();
			Resource pro = sol.getResource("p");
			Resource cls = sol.getResource("c");
			Resource dir = sol.getResource("pDirection");
			String proURI = pro.getURI();
			String clsURI = cls.getURI();
			String dirURI = dir.getURI();

// 			System.out.println(pro.getURI() + " " + cls.getURI() + " " + dir.getURI());
			
			
			Direction direction = null;
			if( dirURI.equals("http://www.w3.org/2000/01/rdf-schema#domain") || dirURI.equals("rdfs:domain")){
				direction = Direction.reverse;
			}
			if( dirURI.equals("http://www.w3.org/2000/01/rdf-schema#range") || dirURI.equals("rdfs:range")){
				if( direction != null ){
					direction = Direction.both;
				}else{
				direction = Direction.forward;
				}
			}
//			System.out.println(direction);
			int numOfLinks = sol.getLiteral("numOfLinks").getInt();
//	System.out.println(numOfLinks);
			ClassLink cl = new ClassLink(proURI, clsURI, direction, numOfLinks);
			solCLs.add(cl);
		}
		qexec.close();
		return solCLs.toArray(new ClassLink[0]);
	}
	
	
	/**
	 *　指定されたクラスを起点とし、そのクラスに属しているインスタンスとリンクが張られているインスタンスの集合を取得し、取得したインスタンスのクラスを網羅的に取得する
	 * <p>
	 * ここでインスタンスとは、rdf:typeの主語として記述されているものをいう
	 * <br>
	 * 処理対象データをgraphURIsで指定することができる
	 * <br>
	 * </p>
	 * 
	 * @param graphURIs　検索対象のgraphのURI配列 　(nullや長さ0の配列も可)
	 * @param originClass 起点となるクラスのURI　（null不可）
	 * @param limit 解として返されるClassLink配列の長さの上限値を指定する。(上限値を設定しない場合は0以下の値を指定する)
	 * @return ClassLinkの配列。ここには、取得されたクラスと関係づけられているプロパティ、関係の向きが含まれる。
	 * @throws Exception
	 * @since 28.01.2014
	 */
	public ClassLink[] getNextClassViaInstanceLink(String[] graphURIs, String originClass,
			int limit) throws Exception {
		StringBuffer queryStr = new StringBuffer();
		queryStr.append("PREFIX owl: <http://www.w3.org/2002/07/owl#>\n");
		queryStr.append("PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>\n");
		queryStr.append("PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#>\n");
		queryStr.append("SELECT DISTINCT ?pf ?pr ?c \n");
		if (graphURIs != null) {
			for (String graphURI : graphURIs) {
				queryStr.append("FROM <");
				queryStr.append(graphURI);
				queryStr.append(">\n");
			}
		}
		queryStr.append("WHERE{\n");
		queryStr.append("  ?ins rdf:type ?c.\n");
		queryStr.append("  ?insOrg rdf:type <");
		queryStr.append(originClass);
		queryStr.append(">.\n");
		queryStr.append("  { ?ins ?pr ?insOrg. }\n  UNION { ?insOrg ?pf ?ins. }\n");
		queryStr.append("}\n");
		if (limit > 0) {
			queryStr.append("limit ");
			queryStr.append(limit);
			queryStr.append("\n");
		}
		
//		System.out.println(queryStr.toString());	
	
		Query query = QueryFactory.create(queryStr.toString());
		QueryExecution qexec = QueryExecutionFactory.sparqlService(endpointURI,
				query);

		ResultSet results = qexec.execSelect();
		ArrayList<ClassLink> solCLs = new ArrayList<ClassLink>();
		for (; results.hasNext();) {
			QuerySolution sol = results.next();
			Resource proForward = sol.getResource("pf");
			Resource proReverse = sol.getResource("pr");
			Resource cls = sol.getResource("c");
			Direction direction = null;
			String propURI = null;
			if( proForward != null ){
				if( proReverse != null ){
					direction = Direction.both;
				}else{
					direction = Direction.forward;
				}
				propURI = proForward.getURI();
			}else{
				direction = Direction.reverse;
				propURI = proReverse.getURI();
			}
			String clsURI = cls.getURI();
//			System.out.println(propURI + " " + clsURI + " " + direction);
			
			ClassLink cl = new ClassLink(propURI, clsURI, direction, 0);
			solCLs.add(cl);
		}
		qexec.close();
		return solCLs.toArray(new ClassLink[0]);
	}

	
	
	public Path[] getPaths(String startClass, String endClass) throws Exception {
		OWLClassGraph graph = new OWLClassGraph(startClass, endClass);
		return graph.getPaths(this);
	}

	public String createSPARQL(Path path) throws Exception {
		return null;
	}


	
	
/*
	private String executeSelect(String sparqlQuery) throws Exception {
		HttpClient client = HttpClientBuilder.create().build();
		HttpPost httppost = new HttpPost(endpointURI);
		httppost.setHeader("Content-Type", "application/x-www-form-urlencoded");
		httppost.setHeader("Accept", "application/sparql-results+xml");
		List<NameValuePair> nvpList = new ArrayList<NameValuePair>();
		nvpList.add(new BasicNameValuePair("query", sparqlQuery));
		// nvpList.add(new BasicNameValuePair("format",
		// outputFormat.getMime()));
		httppost.setEntity(new UrlEncodedFormEntity(nvpList, Charset
				.forName("UTF-8")));
		HttpResponse response = client.execute(httppost);
		// System.out.println("[StatusLine] " + response.getStatusLine());
		HttpEntity entity = response.getEntity();
		String entityString = EntityUtils.toString(entity, "UTF-8");
		return entityString;
	}
*/
	
	/**
	 *　指定されたインスタンスを起点とし、そのインスタンスにリンクが張られているインスタンスの集合を取得する。
	 *　取得された各インスタンスのクラスも網羅的に取得する
	 * <p>
	 * ここでインスタンスとは、rdf:typeの主語として記述されているものをいう
	 * <br>
	 * 処理対象データをgraphURIsで指定することができる
	 * <br>
	 * </p>
	 * 
	 * @param graphURIs　検索対象のgraphのURI配列 　(nullや長さ0の配列も可)
	 * @param originInstance 起点となるインスタンスのURI　（null不可）
	 * @param limit 解として返されるInstanceLink配列の長さの上限値を指定する。(上限値を設定しない場合は0以下の値を指定する)
	 * @return InstanceLinkの配列。ここには、取得されたクラスと関係づけられているプロパティ、関係の向きが含まれる。
	 * @throws Exception
	 * @since 28.01.2014
	 */
	public InstanceLink[] getNextInstancesViaInstanceLink(String[] graphURIs, String originInstance,
			int limit) throws Exception {
		StringBuffer queryStr = new StringBuffer();
		queryStr.append("PREFIX owl: <http://www.w3.org/2002/07/owl#>\n");
		queryStr.append("PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>\n");
		queryStr.append("PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#>\n");
		queryStr.append("SELECT DISTINCT ?pf ?pr ?ins ?c \n");
		if (graphURIs != null) {
			for (String graphURI : graphURIs) {
				queryStr.append("FROM <");
				queryStr.append(graphURI);
				queryStr.append(">\n");
			}
		}
		queryStr.append("WHERE{\n");
//		queryStr.append("  { ?c rdf:type rdfs:Class. }\n  UNION\n  { ?c rdf:type owl:Class. }\n");
		queryStr.append("  ?ins rdf:type ?c.\n");
		queryStr.append("  { ?ins ?pr <");
		queryStr.append(originInstance);
		queryStr.append(">. }\n  UNION { <");
		queryStr.append(originInstance);
		queryStr.append("> ?pf ?ins. }\n");

		queryStr.append(" FILTER (?c != rdf:Property)");

		queryStr.append("}\n");
		if (limit > 0) {
			queryStr.append("limit ");
			queryStr.append(limit);
			queryStr.append("\n");
		}
		
//		System.out.println(queryStr.toString());	
	
		Query query = QueryFactory.create(queryStr.toString());
		QueryExecution qexec = QueryExecutionFactory.sparqlService(endpointURI,
				query);

		ResultSet results = qexec.execSelect();
		HashMap<String, InstanceLink> insLinkMap = new HashMap<String, InstanceLink>();
		for (; results.hasNext();) {
			QuerySolution sol = results.next();
			Resource proForward = sol.getResource("pf");
			Resource proReverse = sol.getResource("pr");
			Resource ins = sol.getResource("ins");
			Resource cls = sol.getResource("c");
			Direction direction = null;
			String propURI = null;
			if( proForward != null ){
				if( proReverse != null ){
					direction = Direction.both;
				}else{
					direction = Direction.forward;
				}
				propURI = proForward.getURI();
			}else{
				direction = Direction.reverse;
				propURI = proReverse.getURI();
			}
			String clsURI = cls.getURI();
			String insURI = ins.getURI();
			String key = propURI + "\t" + insURI + "\t" + direction;
			if( insLinkMap.containsKey(key)){
				InstanceLink insLink = insLinkMap.get(key);
				insLink.addLinkedClassURI(clsURI);
			}else{
				InstanceLink insLink = new InstanceLink(propURI, insURI, new String[]{clsURI}, direction);
				insLinkMap.put(key, insLink);
			}
		}
		qexec.close();
		Collection<InstanceLink> values = insLinkMap.values();
		return values.toArray(new InstanceLink[0]);
	}

}
